/*
 * @(#)Bitmap.java  1.2  2005-07-10
 *
 * Copyright (c) 1999-2005 Werner Randelshofer
 * Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * All rights reserved.
 *
 * This software is the confidential and proprietary information of
 * Werner Randelshofer. ("Confidential Information").  You shall not
 * disclose such Confidential Information and shall use it only in
 * accordance with the terms of the license agreement you entered into
 * with Werner Randelshofer.
 */
package ch.randelshofer.gui.image;

import java.awt.image.ColorModel;
import java.awt.image.IndexColorModel;
import java.awt.image.DirectColorModel;
import java.util.Hashtable;

/**
 * A Bitmap is comprised of a ColorModel and an accessible byte array of
 * image data.
 * <p>
 * The image data is expressed in several layers of rectangular regions
 * called bit-planes. To determine the bits that form a single pixel one
 * must combine all data-bits at the same x,y position in each bit-plane.
 * This is known as a "planar" storage layout as it was used on Commodore
 * Amiga Computers.
 * <p>
 * The bit-planes can be stored contiguously or can be interleaved at each
 * scanline of the image.
 * <p>
 * <p>
 * Fig 1. A sample image:
 * <p><pre>
 * .+++..@...@.+..###...+++.     This sample uses 4 colors:
 * +...+.@@.@@.+.#.....+...+     . = color 0 (all bits clear)
 * +++++:@.@.@.+.#..##.+++++     + = color 1 (bit 0 set, bit 1 clear)
 * +...+.@...@.+.#...#.+...+     @ = color 2 (bit 0 clear, bit 1 set)
 * +...+.@...@.+..####.+...+     # = color 3 (all bits set)
 * </pre><p>
 * Fig 2. Contiguous bit-plane storage layout.
 * <p><pre>
 * 01110000 00001001 11000111 0.......     This is the first bit-plane.
 * 10001000 00001010 00001000 1.......     Each number represents a bit
 * 11111000 00001010 01101111 1.......     in the storage layout. Eight
 * 10001000 00001010 00101000 1.......     bits are grouped into one byte.
 * 10001000 00001001 11101000 1.......     Dots indicate unused bits.
 * <p>
 * 00000010 00100001 11000000 0.......     This is the second bit-plane.
 * 00000011 01100010 00000000 0.......
 * 00000010 10100010 01100000 0.......
 * 00000010 00100010 00100000 0.......
 * 00000010 00100001 11100000 0.......
 * <p></pre>
 * Fig 3. Interleaved bit-plane storage layout.
 * <p><pre>
 * 01110000 00001001 11000111 0.......     This is the first bit-plane.
 * 00000010 00100001 11000000 0.......     This is the second bit-plane.
 * <p>
 * 10001000 00001010 00001000 1.......     The bit-planes are interleaved
 * 00000011 01100010 00000000 0.......     at every scanline of the image.
 * <p>
 * 11111000 00001010 01101111 1.......
 * 00000010 10100010 01100000 0.......
 * <p>
 * 10001000 00001010 00101000 1.......
 * 00000010 00100010 00100000 0.......
 * <p>
 * 10001000 00001001 11101000 1.......
 * 00000010 00100001 11100000 0.......
 * <p></pre>
 * For more details refer to "Amiga ROM Kernel Reference Manual: Libraries,
 * Addison Wesley"
 * <p>
 * <b>Responsibility</b>
 * <p>
 * Gives clients direct access to the image data of the bitmap.
 * Knows how to convert the bitmap into chunky image data according
 * to the current color model.
 * Supports indexed color model, direct color model, 6 and 8 bit HAM color model.
 *
 * @author  Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
 * @version 1.2 2005-07-10 Methods getPixelType() and getChunkyColorModel
 * can now be invoked before a pixel conversion has been done.
 * <br>1.1.1 2004-05-18 Fixed a bug, which caused an image to be all
 * transparent, when it was of bitmap type indexed color, and when the desired
 * bitmap type was true color, and the bitmap had a transparent color.
 * <br>1.1 2003-04-01 Bitmap can now convert bitmaps with IndexColorModel's
 * into chunky pixels with DirectColorModel.
 * <br>1.0  1999-10-19
 */
public class Bitmap
implements Cloneable {
    
    /**  The bitmap data array. */
    private byte[] bitmap;
    
    /** The width of the image. */
    private int width_;
    
    /** The height of the image. */
    private int height_;
    
    /** The number of bits that form a single pixel. */
    private int depth_;
    
    /** BitmapStride is the number of data array elements
     * between two bits of the same image pixel. */
    private int bitplaneStride_;
    
    /** ScanlineStride is the number of data array elements
     * between a given  pixel and the pixel in the same column of
     * the next scanline. */
    private int scanlineStride_;
    
    /** This ColorModel is used for the next conversion from planar
     * bitmap data into chunky pixel data.
     */
    private ColorModel planarColorModel_;
    
    /** This ColorModel represents the preferred color model for chunky pixel.
     * If this value is null, then convertToChunky uses the planarColorModel_.
     */
    private ColorModel preferredChunkyColorModel_;
    
    /** This ColorModel represents the current color model for chunky pixel.
     */
    private ColorModel currentChunkyColorModel_;
    
    /** This ColorModel was used at the previous conversion from
     * planar bitmap into chunky pixel data.
     */
    private ColorModel lastPixelColorModel_;
    
    /** Indicates availability of chunky pixel data. */
    private int pixelType_;
    
    /** Tag for byte pixel data. */
    public final static int BYTE_PIXEL = 1;
    
    /** Tag for integer pixel data. */
    public final static int INT_PIXEL = 2;
    
    /** Tag indicating that no pixel data is available. */
    public final static int NO_PIXEL = 0;
    
    /** Output array for byte pixel data. */
    private byte[] bytePixels_;
    
    /** Output array for integer pixel data. */
    private int[] intPixels_;
    
    /**
     * RGB default color model of this bitmap.
     */
    private final static ColorModel rgbDefault = new DirectColorModel(24, 0xff0000, 0xff00, 0xff);
    
    /**
     * If this boolean is set to true, then convertToChunky always generates
     * chunky pixels using a DirectColorModel.
     */
    private boolean enforceDirectColors_ = false;
    /**
     * If you set this to true, then convertToChunky always generates
     * chunky pixels using a DirectColorModel.
     */
    public void setEnforceDirectColors(boolean b) {
        enforceDirectColors_ = b;
    }
    /**
     * If this returns true, then convertToChunky always generates
     * chunky pixels using a DirectColorModel.
     */
    public boolean isEnforceDirectColors() {
        return enforceDirectColors_;
    }
    
    /**
     * Construct an interleaved bitmap with the specified size,
     * depth and color model.
     * BitplaneStride and ScanlineStride are rounded up to the next
     * even number of bytes.
     * <p>
     * Pre condition:
     *   -
     * <p>
     * Post condition:
     *   Interleaved bitmap constructed.
     * <p>
     * Obligation:
     *   -
     *
     * @param  width  Width in pixels.
     * @param  height  Height in pixels.
     * @param  depth  Number of bits per pixel.
     * @param  colorModel  Color model to be used for conversions from/to chunky pixels.
     */
    public Bitmap(int width, int height, int depth, ColorModel colorModel) {
        this(width, height, depth, colorModel, true);
    }
    
    /**
     * Construct a bitmap with the specified size, depth and color model
     * and with optional interleave.
     * BitplaneStride and ScanlineStride are rounded up to the next
     * even number of bytes.
     * <p>
     * Pre condition:
     *   -
     * <p>
     * Post condition:
     *   Bitmap constructed.
     * <p>
     * Obligation:
     *   -
     *
     * @param  width  Width in pixels.
     * @param  height  Height in pixels.
     * @param  depth  Number of bits per pixel.
     * @param  colorModel  Color model to be used for conversions from/to chunky pixels.
     * @param  isInterleaved  Indicator for contiguous or interleaved bit-planes.
     */
    public Bitmap(int width, int height, int depth, ColorModel colorModel, boolean isInterleaved) {
        this.width_ = width;
        this.height_ = height;
        this.depth_ = depth;
        this.planarColorModel_ = colorModel;
        if (isInterleaved) {
            bitplaneStride_ = (width_ + 15) / 16 * 2;
            scanlineStride_ = bitplaneStride_ * depth_;
            bitmap = new byte[scanlineStride_ * height_];
        } else {
            scanlineStride_ = (width_ + 15) / 16 * 2;
            bitplaneStride_ = scanlineStride_ * depth_;
            bitmap = new byte[bitplaneStride_ * height_];
        }
        pixelType_ = NO_PIXEL;
    }
    
    /**
     * Construct a bitmap with the specified size, depth, color model and
     * interleave.
     * <p>
     * Pre condition:
     * ScanlineStride must be a multiple of BitplaneStride or vice versa.
     * <p>
     * Post condition:
     * Bitmap constructed.
     * <p>
     * Obligation:
     *   -
     *
     * @param  width  Width in pixels.
     * @param  height  Height in pixels.
     * @param  depth  Number of bits per pixel.
     * @param  colorModel  Color model to be used for conversions from/to chunky pixels.
     * @param  bitStride  Number of data array elements between two bits of the same image pixel.
     * @param  scanlineStride  Number of data array elements between a given pixel and the pixel in the same column of
     * the next scanline.
     */
    public Bitmap(int width, int height, int depth, ColorModel colorModel, int bitStride, int scanlineStride) {
        this.width_ = width;
        this.height_ = height;
        this.depth_ = depth;
        this.planarColorModel_ = colorModel;
        this.bitplaneStride_ = bitStride;
        this.scanlineStride_ = scanlineStride;
        if (bitplaneStride_ < scanlineStride_) {
            bitmap = new byte[scanlineStride_ * height_];
        } else {
            bitmap = new byte[bitplaneStride_ * height_];
        }
        pixelType_ = NO_PIXEL;
    }
    
    /**
     * Returns the width of the image.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The width in pixels.
     */
    public int getWidth() {
        return width_;
    }
    
    /**
     * Returns the height of the image.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The height in pixels.
     */
    public int getHeight() {
        return height_;
    }
    
    /**
     * Returns the depth of the image.
     * <p>
     * The depth indicates how many bits are used to form a single pixel.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The number of bitplanes used to form a single pixel.
     */
    public int getDepth() {
        return depth_;
    }
    
    /**
     * Returns the numer of bytes you must add to a given address
     * in the bitmap to advance to the next scanline of the image.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The scansize.
     */
    public int getScanlineStride() {
        return scanlineStride_;
    }
    
    /**
     * Returns the number of bytes that you must add to a bitmap address
     * to advance to the next bit of a scanline.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The interleave of the bitmap.
     */
    public int getBitplaneStride() {
        return bitplaneStride_;
    }
    
    /**
     * Replaces the color model used for conversions from/to chunky pixels.
     * <p>
     * Pre condition: The new color model must correspond with the depth of the bitmap.
     * <p>
     * Post condition: Color model changed.
     * <p>
     * Obligation: -
     *
     * @param  The new color model.
     */
    public void setPlanarColorModel(ColorModel colorModel) {
        planarColorModel_ = colorModel;
    }
    
    /**
     * Returns the current color model of the planar image in this bitmap.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The color model.
     */
    public ColorModel getPlanarColorModel() {
        return planarColorModel_;
    }
    /**
     * Sets the preferred color model used for to chunky pixels.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: Color model changed.
     * <p>
     * Obligation: -
     *
     * @param  The new color model. Specify null to let the bitmap choose
     * the color model on its own.
     */
    public void setPreferredChunkyColorModel(ColorModel colorModel) {
        preferredChunkyColorModel_ = colorModel;
    }
    /**
     * Returns the preferred color model used for to chunky pixels.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -.
     * <p>
     * Obligation: -
     *
     * @return  The new color model. Returns null, if the preferred color model
     * shall be chosen based on the planar color model.
     */
    public ColorModel getPreferredChunkyColorModel() {
        return preferredChunkyColorModel_;
    }
    
    /**
     * Returns the current color model of the chunky image in this bitmap.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  The color model.
     */
    public ColorModel getChunkyColorModel() {
        if (currentChunkyColorModel_ == null) {
            if (planarColorModel_ instanceof HAMColorModel) {
                currentChunkyColorModel_ = planarColorModel_;
            } else if (planarColorModel_ instanceof IndexColorModel) {
                if (enforceDirectColors_
                || preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault)) {
                    currentChunkyColorModel_ = (preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault))
                    ? preferredChunkyColorModel_
                    : rgbDefault;
                } else {
                    currentChunkyColorModel_ = planarColorModel_;
                }
            } else if (planarColorModel_ instanceof DirectColorModel) {
                currentChunkyColorModel_ = (preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault))
                ? preferredChunkyColorModel_
                : rgbDefault;
            } else {
                throw new InternalError("unsupported color model:" + planarColorModel_);
            }
        }
        return currentChunkyColorModel_;
    }
    
    /**
     * Gives you direct access to the bitmap data array.
     * <p>
     * Pre condition: -.
     * <p>
     * Post condition: -
     * <p>
     * Obligation: The bitmap data array remains property
     * of the Bitmap and will be used at the next
     * conversion to chunky. You can access it as you
     * like (even during conversion) since this class
     * does never change the contents of the bitmap.
     *
     * @return  A reference to the bitmap data.
     */
    public byte[] getBitmap() {
        return bitmap;
    }
    
    /**
     * Returns a reference to the byte pixel data that has been
     * generated by a previous call to #converToChunky.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: You may modify the contents of the array
     * as you like to get some nice effects for the
     * next call to #convertToChunky. Note whovewer that
     * #convertToChunky will not reuse this array when
     * the colorModel has been changed to a color format
     * that requires pixels in integer format.
     *
     * @return  byte array or NULL when no byte pixels have been
     * generated by #convertToChunky.
     */
    public byte[] getBytePixels() {
        if (pixelType_ == BYTE_PIXEL) {
            return bytePixels_;
        } else {
            return null;
        }
    }
    
    /**
     * Returns a reference to the integer pixel data that has been
     * generated by a previous call to #converToChunky.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: You may modify the contents of the array
     * as you like to get some nice effects for the
     * next call to #convertToChunky. Note however that
     * #convertToChunky will not reuse this array when
     * the colorModel has been changed to a color format
     * that requires pixels in byte format.
     *
     * @return  byte array or NULL when no int pixels have been
     * generated by #convertToChunky.
     */
    public int[] getIntPixels() {
        if (pixelType_ == INT_PIXEL) {
            return intPixels_;
        } else {
            return null;
        }
    }
    
    /**
     * Returns the available type of pixel data.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: -
     * <p>
     * Obligation: -
     *
     * @return  A constant that specifies the current type of pixel data.
     */
    public int getPixelType() {
        if (pixelType_ == NO_PIXEL) {
            if (planarColorModel_ instanceof HAMColorModel) {
                pixelType_ = INT_PIXEL;
            } else if (planarColorModel_ instanceof IndexColorModel) {
                if (enforceDirectColors_
                || preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault)) {
                    pixelType_ = INT_PIXEL;
                } else {
                    pixelType_ = BYTE_PIXEL;
                }
            } else if (planarColorModel_ instanceof DirectColorModel) {
                pixelType_ = INT_PIXEL;
            } else {
                throw new InternalError("unsupported color model:" + planarColorModel_);
            }
        }
        return pixelType_;
    }
    
    
    /**
     * Creates a clone.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: Clone created.
     *
     * @return  A clone.
     */
    public Object clone() {
        try {
            Bitmap theClone = (Bitmap)super.clone();
            theClone.bitmap = (byte[])bitmap.clone();
            if (getPixelType() == BYTE_PIXEL) {
                theClone.bytePixels_ = (byte[])bytePixels_.clone();
            }
            if (getPixelType() == INT_PIXEL) {
                theClone.intPixels_ = (int[])intPixels_.clone();
            }
            return theClone;
        } catch (CloneNotSupportedException e) {
            throw new InternalError(e.toString());
        }
    }
    
    /**
     * Converts the planar image data into chunky pixel data.
     * <p>
     * This method will either generate byte pixel data or integer
     * pixel data (depending on the color model).
     * <p>
     * The pixel array that resulted to a prior call to this
     * method will be reused when the image dimension and the color
     * model allows for it.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: Chunky pixels generated.
     * <p>
     * Obligation: -
     *
     * @return The type of generated pixel data.
     */
    public int convertToChunky() {
        return convertToChunky(0, 0, getHeight() - 1, getWidth() - 1);
    }
    
    /**
     * Converts the indicated area of the bitmap data into  pixel data.
     * <p>
     * This method will either generate byte pixel data or integer
     * pixel data (depending on the color model).
     * <p>
     * Note that the size of the generated pixel data always corresponds
     * to the size of the complete image. You do only specify a subset
     * of the image to be <i>converted</i> not a subset to be extracted.
     * Note also that the pixel data that resulted from prior calls to
     * this method will be reused when the generated pixel array was
     * of the same size and type.
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: The indicated part of the bitmap has been
     *   converted into chunky pixels.
     * <p>
     * Obligation: -
     *
     * @return The type of generated pixel data.
     */
    public int convertToChunky(int top,int left,int bottom,int right) {
        //long start = System.currentTimeMillis();
        pixelType_ = NO_PIXEL;
        
        /* Ensure pre conditions are met. */
        if (top < 0) {
            top = 0;
        }
        if (left < 0) {
            left = 0;
        }
        if (bottom > getHeight() - 1) {
            bottom = getHeight() - 1;
        }
        if (right > getWidth() - 1) {
            right = getWidth() - 1;
        }
        
        /* */
        if (planarColorModel_ instanceof HAMColorModel) {
            if (intPixels_ == null || intPixels_.length != getWidth()*getHeight()) {
                bytePixels_ = null;
                intPixels_ = new int[getWidth()*getHeight()];
            }
            currentChunkyColorModel_ = planarColorModel_;
            if (((HAMColorModel)planarColorModel_).getHAMType() == HAMColorModel.HAM6) {
                ham6PlanesToDirectPixels(top,left,bottom,right);
            } else if (((HAMColorModel)planarColorModel_).getHAMType() == HAMColorModel.HAM8) {
                ham8PlanesToDirectPixels(top,left,bottom,right);
            } else {
                throw new InternalError("unsupported ham model:" + planarColorModel_);
            }
            pixelType_ = INT_PIXEL;
            
        } else if (planarColorModel_ instanceof IndexColorModel) {
            if (enforceDirectColors_
            || preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault)) {
                if (intPixels_ == null || intPixels_.length != getWidth()*getHeight()) {
                    bytePixels_ = null;
                    intPixels_ = new int[getWidth()*getHeight()];
                }
                currentChunkyColorModel_ = (preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault))
                ? preferredChunkyColorModel_
                : rgbDefault;
                
                indexPlanesToDirectPixels(top, left, bottom, right);
                pixelType_ = INT_PIXEL;
            } else {
                if (bytePixels_ == null || bytePixels_.length != getWidth()*getHeight()) {
                    intPixels_ = null;
                    bytePixels_ = new byte[getWidth()*getHeight()];
                }
                currentChunkyColorModel_ = planarColorModel_;
                indexPlanesToIndexPixels(top,left,bottom,right);
                pixelType_ = BYTE_PIXEL;
            }
        } else if (planarColorModel_ instanceof DirectColorModel) {
            if (intPixels_ == null || intPixels_.length != getWidth()*getHeight()) {
                bytePixels_ = null;
                intPixels_ = new int[getWidth()*getHeight()];
            }
            currentChunkyColorModel_ = (preferredChunkyColorModel_ != null && preferredChunkyColorModel_.equals(rgbDefault))
            ? preferredChunkyColorModel_
            : rgbDefault;
            
            directPlanesToDirectPixels(top, left, bottom, right);
            pixelType_ = INT_PIXEL;
        } else {
            throw new InternalError("unsupported color model:" + planarColorModel_);
        }
        return pixelType_;
    }
    
    /**
     * Frees the memory allocated for the pixel data.
     *
     * <p>
     * Pre condition: -
     * <p>
     * Post condition: The bitmap has given up all its
     * references to the pixel data.
     * <p>
     * Obligation: The pixel data will not be reused at the
     * next call to #convertToChunky.
     */
    public void flushPixels() {
        pixelType_ = NO_PIXEL;
        intPixels_ = null;
        bytePixels_ = null;
    }
    
    /**
     * Converts the planar image data into chunky pixels.
     *
     * After successful completion the chunky pixels can by used
     * in conjunction with the IndexColorModel associated to
     * this instance.
     *
     * Pre condition
     *   The color model must be an instance of java.awt.IndexColorModel.
     *   0 <= topBound <= bottomBound <= height.
     *   0 <= leftBound <= rightBound <= width.
     * Post condition
     *   -
     * Obligation
     *   -
     *
     * @author  Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
     * @version  1997-10-16  Created.
     */
    private void indexPlanesToIndexPixels(int top, int left, int bottom, int right) {
        //long startTime = System.currentTimeMillis();
        
        /* Add one to bottom and right to facilitate computations. */
        bottom++;
        right++;
        
        final int scanlineStride = getScanlineStride();
        final int bitplaneStride = getBitplaneStride();
        final int depth = getDepth();
        final int width = getWidth();
        final int pixelLineStride = width - right + left;
        final int bottomScanline = bottom * scanlineStride;
        //final int bitCorrection = depth - 8;
        //final int bitCorrection = 8 - depth;
        int x;
        int iPixel = top * width + left;
        int pixel = 0;
        //int bitShift;
        int iBitmap;
        int iScanline;
        int iDepth;
        int b0,b1,b2,b3,b4,b5,b6,b7;
        b0=b1=b2=b3=b4=b5=b6=b7=0;
        final int bitplaneStride1 = bitplaneStride;
        final int bitplaneStride2 = bitplaneStride * 2;
        final int bitplaneStride3 = bitplaneStride * 3;
        final int bitplaneStride4 = bitplaneStride * 4;
        final int bitplaneStride5 = bitplaneStride * 5;
        final int bitplaneStride6 = bitplaneStride * 6;
        final int bitplaneStride7 = bitplaneStride * 7;
        
        int iBit; // the index of the bit inside the byte at the current x-position
        int bitMask; // the mask for the bit inside the byte at the current x-position
        
        switch (depth) {
            case 1 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        bytePixels_[iPixel++] = (byte) (((bitmap[iScanline + (x >>> 3)] << (x & 7)) & 128) >>> 7);
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 2 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        bytePixels_[iPixel++] = (byte) ((
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 3 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        bytePixels_[iPixel++] = (byte) ((
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        | (bitmap[iBitmap+bitplaneStride2] & bitMask) << 2
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 4 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        bytePixels_[iPixel++] = (byte) ((
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        | (bitmap[iBitmap+bitplaneStride2] & bitMask) << 2
                        | (bitmap[iBitmap+bitplaneStride3] & bitMask) << 3
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 5 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                        }
                        bytePixels_[iPixel++] = (byte) ((
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 6 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                        }
                        bytePixels_[iPixel++] = (byte) ((
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 7 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                            b6 = bitmap[iBitmap+bitplaneStride6];
                        }
                        bytePixels_[iPixel++] = (byte) ((
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        | (b6 & bitMask) << 6
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 8 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                            b6 = bitmap[iBitmap+bitplaneStride6];
                            b7 = bitmap[iBitmap+bitplaneStride7];
                        }
                        bytePixels_[iPixel++] = (byte) ((
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        | (b6 & bitMask) << 6
                        | (b7 & bitMask) << 7
                        ) >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            default :
                for (iScanline = top * scanlineStride + scanlineStride; iScanline <= bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        pixel = 0;
                        for (iDepth = 0; iDepth < depth; iDepth++) {
                            iBitmap -= bitplaneStride;
                            pixel = (pixel << 1) | bitmap[iBitmap] & bitMask;
                        }
                        bytePixels_[iPixel++] = (byte) (pixel >>> (7 - iBit));
                    }
                    iPixel += pixelLineStride;
                }
        }
        //System.out.println((System.currentTimeMillis() - startTime) + " " + depth);
    }
    
    /**
     * Converts the planar image data into chunky pixels.
     *
     * After successful completion the chunky pixels can by used
     * in conjunction with the DirectColorModel associated to
     * this instance.
     *
     * Pre condition
     *   The color model must be an instance of java.awt.IndexColorModel.
     *   0 <= topBound <= bottomBound <= height.
     *   0 <= leftBound <= rightBound <= width.
     * Post condition
     *   -
     * Obligation
     *   -
     */
    private void indexPlanesToDirectPixels(int top, int left, int bottom, int right) {
        //long startTime = System.currentTimeMillis();
        IndexColorModel colorModel = (IndexColorModel) planarColorModel_;
        final int[] clut = new int[colorModel.getMapSize()];
        colorModel.getRGBs(clut);
        if (clut.length < (1 << getDepth())) throw new IndexOutOfBoundsException("Clut must not be smaller than depth");
        
        /* Add one to bottom and right to facilitate computations. */
        bottom++;
        right++;
        
        final int scanlineStride = getScanlineStride();
        final int bitplaneStride = getBitplaneStride();
        final int depth = getDepth();
        final int width = getWidth();
        final int pixelLineStride = width - right + left;
        final int bottomScanline = bottom * scanlineStride;
        //final int bitCorrection = 8 - depth;
        int x;
        int iPixel = top * width + left;
        int pixel = 0;
        //int bitShift;
        //int iBitmap;
        int iScanline;
        int iDepth;
        
        
        int iBit; // the index of the bit inside the byte at the current x-position
        int bitMask; // the mask for the bit inside the byte at the current x-position
        
        final int bitplaneStride1 = bitplaneStride;
        final int bitplaneStride2 = bitplaneStride * 2;
        final int bitplaneStride3 = bitplaneStride * 3;
        final int bitplaneStride4 = bitplaneStride * 4;
        final int bitplaneStride5 = bitplaneStride * 5;
        final int bitplaneStride6 = bitplaneStride * 6;
        final int bitplaneStride7 = bitplaneStride * 7;
        
        int iBitmap;
        int b0, b1, b2, b3, b4, b5, b6, b7;
        b0 = b1 = b2 = b3 = b4 = b5 = b6 = b7 = 0;
        
        switch (depth) {
            case 1 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        intPixels_[iPixel++] = clut[(((bitmap[iScanline + (x >>> 3)] << (x & 7)) & 128) >>> 7)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 2 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        intPixels_[iPixel++] = clut[(
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 3 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        intPixels_[iPixel++] = clut[(
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        | (bitmap[iBitmap+bitplaneStride2] & bitMask) << 2
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 4 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        intPixels_[iPixel++] = clut[(
                        (bitmap[iBitmap] & bitMask)
                        | (bitmap[iBitmap+bitplaneStride1] & bitMask) << 1
                        | (bitmap[iBitmap+bitplaneStride2] & bitMask) << 2
                        | (bitmap[iBitmap+bitplaneStride3] & bitMask) << 3
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 5 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                        }
                        intPixels_[iPixel++] = clut[(
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                
                break;
                
            case 6 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                        }
                        intPixels_[iPixel++] = clut[(
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 7 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                            b6 = bitmap[iBitmap+bitplaneStride6];
                        }
                        intPixels_[iPixel++] = clut[(
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        | (b6 & bitMask) << 6
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            case 8 :
                for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        
                        if (iBit == 0) {
                            b0 = bitmap[iBitmap];
                            b1 = bitmap[iBitmap+bitplaneStride];
                            b2 = bitmap[iBitmap+bitplaneStride2];
                            b3 = bitmap[iBitmap+bitplaneStride3];
                            b4 = bitmap[iBitmap+bitplaneStride4];
                            b5 = bitmap[iBitmap+bitplaneStride5];
                            b6 = bitmap[iBitmap+bitplaneStride6];
                            b7 = bitmap[iBitmap+bitplaneStride7];
                        }
                        intPixels_[iPixel++] = clut[(
                        (b0 & bitMask)
                        | (b1 & bitMask) << 1
                        | (b2 & bitMask) << 2
                        | (b3 & bitMask) << 3
                        | (b4 & bitMask) << 4
                        | (b5 & bitMask) << 5
                        | (b6 & bitMask) << 6
                        | (b7 & bitMask) << 7
                        ) >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
                break;
                
            default :
                for (iScanline = top * scanlineStride + scanlineStride; iScanline <= bottomScanline; iScanline += scanlineStride) {
                    for (x = left; x < right; x++) {
                        iBit = x & 7;
                        bitMask = 128 >>> (iBit);
                        iBitmap = iScanline + (x >>> 3);
                        pixel = 0;
                        for (iDepth = 0; iDepth < depth; iDepth++) {
                            iBitmap -= bitplaneStride;
                            pixel = (pixel << 1) | bitmap[iBitmap] & bitMask;
                        }
                        intPixels_[iPixel++] =
                        clut[pixel >>> (7 - iBit)];
                    }
                    iPixel += pixelLineStride;
                }
        }
    }
    
    /**
     * Converts the planar image data into chunky pixels.
     *
     * After successful completion the chunky pixels can by used
     * in conjunction with the DirectColorModel associated to
     * this instance.
     *
     * Pre condition
     *   The color model must be an instance of java.awt.DirectColorModel.
     *   0 <= topBound <= bottomBound <= height.
     *   0 <= leftBound <= rightBound <= width.
     * Post condition
     *   -
     * Obligation
     *   -
     */
    private void directPlanesToDirectPixels(int top, int left, int bottom, int right) {
        /* Add one to bottom and right to facilitate computations. */
        bottom++;
        right++;
        
        final int scanlineStride = getScanlineStride();
        final int bitplaneStride = getBitplaneStride();
        final int depth = getDepth();
        final int width = getWidth();
        final int pixelLineStride = width - right + left;
        final int bottomScanline = bottom * scanlineStride;
        int x;
        int iPixel = top * width + left;
        int pixel = 0;
        int bitShift;
        int iScanline;
        int iDepth;
        final int bitplaneStride2 = bitplaneStride * 2;
        final int bitplaneStride3 = bitplaneStride * 3;
        final int bitplaneStride4 = bitplaneStride * 4;
        final int bitplaneStride5 = bitplaneStride * 5;
        final int bitplaneStride6 = bitplaneStride * 6;
        final int bitplaneStride7 = bitplaneStride * 7;
        final int bitplaneStride8 = bitplaneStride * 8;
        final int bitplaneStride9 = bitplaneStride * 9;
        final int bitplaneStride10 = bitplaneStride * 10;
        final int bitplaneStride11 = bitplaneStride * 11;
        final int bitplaneStride12 = bitplaneStride * 12;
        final int bitplaneStride13 = bitplaneStride * 13;
        final int bitplaneStride14 = bitplaneStride * 14;
        final int bitplaneStride15 = bitplaneStride * 15;
        final int bitplaneStride16 = bitplaneStride * 16;
        final int bitplaneStride17 = bitplaneStride * 17;
        final int bitplaneStride18 = bitplaneStride * 18;
        final int bitplaneStride19 = bitplaneStride * 19;
        final int bitplaneStride20 = bitplaneStride * 20;
        final int bitplaneStride21 = bitplaneStride * 21;
        final int bitplaneStride22 = bitplaneStride * 22;
        final int bitplaneStride23 = bitplaneStride * 23;
        
        int iBitmap = top * scanlineStride + left / 8;
        int b0=bitmap[iBitmap];
        int b1=bitmap[iBitmap+bitplaneStride];
        int b2=bitmap[iBitmap+bitplaneStride2];
        int b3=bitmap[iBitmap+bitplaneStride4];
        int b4=bitmap[iBitmap+bitplaneStride4];
        int b5=bitmap[iBitmap+bitplaneStride5];
        int b6=bitmap[iBitmap+bitplaneStride6];
        int b7=bitmap[iBitmap+bitplaneStride7];
        int b8=bitmap[iBitmap+bitplaneStride8];
        int b9=bitmap[iBitmap+bitplaneStride9];
        int b10=bitmap[iBitmap+bitplaneStride10];
        int b11=bitmap[iBitmap+bitplaneStride11];
        int b12=bitmap[iBitmap+bitplaneStride12];
        int b13=bitmap[iBitmap+bitplaneStride13];
        int b14=bitmap[iBitmap+bitplaneStride14];
        int b15=bitmap[iBitmap+bitplaneStride15];
        int b16=bitmap[iBitmap+bitplaneStride16];
        int b17=bitmap[iBitmap+bitplaneStride17];
        int b18=bitmap[iBitmap+bitplaneStride18];
        int b19=bitmap[iBitmap+bitplaneStride19];
        int b20=bitmap[iBitmap+bitplaneStride20];
        int b21=bitmap[iBitmap+bitplaneStride21];
        int b22=bitmap[iBitmap+bitplaneStride22];
        int b23=bitmap[iBitmap+bitplaneStride23];
        
        int iBit, bitMask;
        for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
            for (x = left; x < right; x++) {
                iBit = x & 7;
                bitMask = 128 >>> (iBit);
                iBitmap = iScanline + (x >>> 3);
                if (iBit == 0) {
                    b0 = bitmap[iBitmap];
                    b1 = bitmap[iBitmap+bitplaneStride];
                    b2 = bitmap[iBitmap+bitplaneStride2];
                    b3 = bitmap[iBitmap+bitplaneStride3];
                    b4 = bitmap[iBitmap+bitplaneStride4];
                    b5 = bitmap[iBitmap+bitplaneStride5];
                    b6 = bitmap[iBitmap+bitplaneStride6];
                    b7 = bitmap[iBitmap+bitplaneStride7];
                    b8 = bitmap[iBitmap+bitplaneStride8];
                    b9 = bitmap[iBitmap+bitplaneStride9];
                    b10 = bitmap[iBitmap+bitplaneStride10];
                    b11 = bitmap[iBitmap+bitplaneStride11];
                    b12 = bitmap[iBitmap+bitplaneStride12];
                    b13 = bitmap[iBitmap+bitplaneStride13];
                    b14 = bitmap[iBitmap+bitplaneStride14];
                    b15 = bitmap[iBitmap+bitplaneStride15];
                    b16 = bitmap[iBitmap+bitplaneStride16];
                    b17 = bitmap[iBitmap+bitplaneStride17];
                    b18 = bitmap[iBitmap+bitplaneStride18];
                    b19 = bitmap[iBitmap+bitplaneStride19];
                    b20 = bitmap[iBitmap+bitplaneStride20];
                    b21 = bitmap[iBitmap+bitplaneStride21];
                    b22 = bitmap[iBitmap+bitplaneStride22];
                    b23 = bitmap[iBitmap+bitplaneStride23];
                }
                
                intPixels_[iPixel++] = (
                (b0 & bitMask) << 16
                | (b1 & bitMask) << 17
                | (b2 & bitMask) << 18
                | (b3 & bitMask) << 19
                | (b4 & bitMask) << 20
                | (b5 & bitMask) << 21
                | (b6 & bitMask) << 22
                | (b7 & bitMask) << 23
                | (b8 & bitMask) << 8
                | (b9 & bitMask) << 9
                | (b10 & bitMask) << 10
                | (b11 & bitMask) << 11
                | (b12 & bitMask) << 12
                | (b13 & bitMask) << 13
                | (b14 & bitMask) << 14
                | (b15 & bitMask) << 15
                | (b16 & bitMask)
                | (b17 & bitMask) << 1
                | (b18 & bitMask) << 2
                | (b19 & bitMask) << 3
                | (b20 & bitMask) << 4
                | (b21 & bitMask) << 5
                | (b22 & bitMask) << 6
                | (b23 & bitMask) << 7
                ) >>> (7 - iBit);
                
            }
            iPixel += pixelLineStride;
        }
    }
    
    /**
     * Converts the planar image data into chunky pixels.
     *
     * After successful completion the chunky pixels can by used
     * in conjunction with the HAMColorModel associated to
     * this instance.
     *
     * Pre condition
     *   The color model must be an instance of HAMColorModel.
     *   0 <= topBound <= bottomBound <= height.
     *   0 <= leftBound <= rightBound <= width.
     * Post condition
     *   -
     * Obligation
     *   -
     *
     * @author  Werner Randelshofer, Staldenmattweg 2, CH-6405 Immensee, Switzerland
     * @version  1997-10-16  Created.
     */
    private void ham6PlanesToDirectPixels(int top, int left, int bottom, int right) {
        /* Add one to bottom and right to facilitate computations. */
        bottom++;
        right++;
        
        final int[] HAMColors = new int[((HAMColorModel)planarColorModel_).getMapSize()];
        ((HAMColorModel)planarColorModel_).getRGBs(HAMColors);
        final int scanlineStride = getScanlineStride();
        final int bitplaneStride = getBitplaneStride();
        final int depth = getDepth();
        final int width = getWidth();
        final int pixelLineStride = width - right + left;
        final int bottomScanline = bottom * scanlineStride;
        int x;
        int iPixel = top * width + left;
        int lastPixel, iLastPixel = top*width + left - 1;
        int pixel = 0;
        int bitShift;
        int iScanline;
        int iDepth;
        final int bitplaneStride1 = bitplaneStride;
        final int bitplaneStride2 = bitplaneStride * 2;
        final int bitplaneStride3 = bitplaneStride * 3;
        final int bitplaneStride4 = bitplaneStride * 4;
        final int bitplaneStride5 = bitplaneStride * 5;
        int iBitmap;
        int b0,b1,b2,b3,b4,b5;
        b0=b1=b2=b3=b4=b5=0;
        int iBit; // the index of the bit inside the byte at the current x-position
        int bitMask; // the mask for the bit inside the byte at the current x-position
        
        for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
            if (left == 0) {
                lastPixel = 0xff000000;
            } else {
                lastPixel = intPixels_[iLastPixel];
                iLastPixel += width;
            }
            for (x = left; x < right; x++) {
                iBit = x & 7;
                bitMask = 128 >>> (iBit);
                iBitmap = iScanline + (x >>> 3);
                
                if (iBit == 0) {
                    b0 = bitmap[iBitmap];
                    b1 = bitmap[iBitmap+bitplaneStride];
                    b2 = bitmap[iBitmap+bitplaneStride2];
                    b3 = bitmap[iBitmap+bitplaneStride3];
                    b4 = bitmap[iBitmap+bitplaneStride4];
                    b5 = bitmap[iBitmap+bitplaneStride5];
                }
                pixel = (
                (b0 & bitMask)
                | (b1 & bitMask) << 1
                | (b2 & bitMask) << 2
                | (b3 & bitMask) << 3
                ) >>> (7 - iBit);
                
                switch (
                ((b4 & bitMask)
                | (b5 & bitMask) << 1
                ) >>> (7 - iBit)) {
                    
                    case 0: // use indexed color
                        intPixels_[iPixel++] = lastPixel = HAMColors[pixel];
                        break;
                        
                    case 1: // modifie blue
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xffffff00 | pixel | pixel << 4;
                        break;
                        
                    case 2:  // modify red
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xff00ffff | pixel << 16 | pixel << 20;
                        break;
                        
                    default: // modify green
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xffff00ff | pixel << 8 | pixel << 12;
                        break;
                }
            }
            iPixel += pixelLineStride;
        }
    }
    /**
     * Converts the planar image data into chunky pixels.
     *
     * After successful completion the chunky pixels can by used
     * in conjunction with the HAMColorModel associated to
     * this instance.
     *
     * Pre condition
     *   The color model must be an instance of HAMColorModel.
     *   0 <= topBound <= bottomBound <= height.
     *   0 <= leftBound <= rightBound <= width.
     * Post condition
     *   -
     * Obligation
     *   -
     */
    private void ham8PlanesToDirectPixels(int top, int left, int bottom, int right) {
        /* Add one to bottom and right to facilitate computations. */
        bottom++;
        right++;
        
        final int[] HAMColors = new int[((HAMColorModel)planarColorModel_).getMapSize()];
        ((HAMColorModel)planarColorModel_).getRGBs(HAMColors);
        final int scanlineStride = getScanlineStride();
        final int bitplaneStride = getBitplaneStride();
        final int depth = getDepth();
        final int width = getWidth();
        final int pixelLineStride = width - right + left;
        final int bottomScanline = bottom * scanlineStride;
        int x;
        int iPixel = top * width + left;
        int lastPixel, iLastPixel = top*width + left - 1;
        int pixel = 0;
        int bitShift;
        //int iBitmap;
        int iScanline;
        int iDepth;
        final int bitplaneStride1 = bitplaneStride;
        final int bitplaneStride2 = bitplaneStride * 2;
        final int bitplaneStride3 = bitplaneStride * 3;
        final int bitplaneStride4 = bitplaneStride * 4;
        final int bitplaneStride5 = bitplaneStride * 5;
        final int bitplaneStride6 = bitplaneStride * 6;
        final int bitplaneStride7 = bitplaneStride * 7;
        
        int iBitmap = top * scanlineStride + left / 8;
        int b0,b1,b2,b3,b4,b5,b6,b7;
        b0=b1=b2=b3=b4=b5=b6=b7=0;
        
        int iBit; // the index of the bit inside the byte at the current x-position
        int bitMask; // the mask for the bit inside the byte at the current x-position
        
        for (iScanline = top * scanlineStride; iScanline < bottomScanline; iScanline += scanlineStride) {
            if (left == 0) {
                lastPixel = 0xff000000;
            } else {
                lastPixel = intPixels_[iLastPixel];
                iLastPixel += width;
            }
            for (x = left; x < right; x++) {
                iBit = x & 7;
                bitMask = 128 >>> (iBit);
                iBitmap = iScanline + (x >>> 3);
                
                if (iBit == 0) {
                    b0 = bitmap[iBitmap];
                    b1 = bitmap[iBitmap+bitplaneStride];
                    b2 = bitmap[iBitmap+bitplaneStride2];
                    b3 = bitmap[iBitmap+bitplaneStride3];
                    b4 = bitmap[iBitmap+bitplaneStride4];
                    b5 = bitmap[iBitmap+bitplaneStride5];
                    b6 = bitmap[iBitmap+bitplaneStride6];
                    b7 = bitmap[iBitmap+bitplaneStride7];
                }
                pixel = (
                (b0 & bitMask)
                | (b1 & bitMask) << 1
                | (b2 & bitMask) << 2
                | (b3 & bitMask) << 3
                | (b4 & bitMask) << 4
                | (b5 & bitMask) << 5
                ) >>> (7 - iBit);
                
                switch (
                ((b6 & bitMask)
                | (b7 & bitMask) << 1
                ) >>> (7 - iBit)) {
                    
                    case 0: // use indexed color
                        intPixels_[iPixel++] = lastPixel = HAMColors[pixel];
                        break;
                        
                    case 1: // modifie blue
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xffffff00 | pixel << 2 | pixel >>> 4;
                        break;
                        
                    case 2:  // modify red
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xff00ffff | pixel << 18 | (pixel & 0x03) << 16;
                        break;
                        
                    default: // modify green
                        intPixels_[iPixel++] = lastPixel = lastPixel & 0xffff00ff | pixel << 10 | (pixel & 0x03) << 8;
                        break;
                }
            }
            iPixel += pixelLineStride;
        }
    }
}
